/*
 * File: Wallsaver.js
 *
 * Copyright (c) 2007-2008, Paulo Avila (apaulodesign.com)
 *
 * Project: Wallsaver Widget
 *
 * Description: Provides the necessary functions for the operation of the
 * Wallsaver widget - a wrapper for a command line operation that allows
 * a Screen Saver to be run on the desktop wallpaper.
 *
 * Modification Log:
 *	2007.08.18	Paulo	- Initial setup of functions
 *	          			- v1.0 release
 *	2008.03.12	Paulo	- Modified code to make use of the Widget Resource (wr_*) scripts
 *	2008.09.11	Paulo	- Shows version on back
 *	          			- Able to check for updates via AJAX
 *	2008.09.12	Paulo	- Removes saved preferences when widget closed
 *	          			- If newer version exists, clicking the version again redirects to site
 *	          			- v1.1 release
 *	2008.09.13	Paulo	- Dynamically get list of installed Screen Savers
 *	2008.09.14	Paulo	- Modified URLs to include GET data and make use of the wr_openURL() resource
 *	2008.09.14	Paulo	- Widget-specific function names now have a prefix of "w_"
 *	2008.09.15	Paulo	- Further modularized code for proper functionality in Safari
 *	          			- v1.2 release
 */

var PREF_KEY_SAVER_NAME = "ScreenSaverOfChoice";

var EXTERNAL_URL = "http://www.apaulodesign.com/widgets/index.php";
var VERSION_CHECK_APP = "http://www.apaulodesign.com/widgets/version.php";

var g_ExecButton;	// exec button object
var g_DoneButton;	// done button object
var g_InfoButton;	// info button object

var g_MyCommand = null;										// object returned when the command is run
var g_SaverName = "Default";								// the name of the screen saver that will run
var g_CurrentVersion = wr_getPlistValue("CFBundleVersion");	// current version of widget


/*
 * Initializes global objects, registers the widget handlers and
 * loads all installed screen savers as well as any saved preferences.
 */
function w_init()
{
	w_registerHandlers();
	w_loadScreenSavers();
	w_loadSavedPrefs();		// must come after screen savers have been loaded
}


/*
 * Depending on whether the ScreenSaverEngine is running or not,
 * this function will stop or start it accordingly.
 */
function w_toggleSaver()
{
	if (g_MyCommand)
		w_stopSaverEngine();
	else
		w_runSaverEngine();
}


/*
 * Called before transitioning to the back.
 * Resets the displayed version.
 */
function w_toBack()
{
	document.getElementById("version").style.color = "gray";
	document.getElementById("version").innerHTML = "v" + g_CurrentVersion;
	document.getElementById("version").onclick = wr_versionQuery;

	wr_showBack();
}


/*
 * Called before transitioning to the front.
 */
function w_toFront()
{
	wr_showFront();
}


/*
 * Populates the list of available screen savers
 * by checking 3 locations on the hard drive:
 *
 * ~/Library/Screen Savers
 * /Library/Screen Savers
 * /System/Library/Screen Savers
 */
function w_loadScreenSavers()
{
	var i;
	var whoami;
	var installedSavers;
	var listElement = document.getElementById("screenSaverList");

	if (window.widget)
	{
		whoami = widget.system("/usr/bin/whoami", null).outputString.split("\n")[0];

		// 2D array with [screen saver names][group label]
		installedSavers = new Array(
			new Array(widget.system("/bin/ls '/System/Library/Screen Savers' | grep -v .slideSaver", null).outputString, ""),
			new Array(widget.system("/bin/ls '/Users/" + whoami + "/Library/Screen Savers'", null).outputString, whoami + "'s Screen Savers"),
			new Array(widget.system("/bin/ls '/System/Library/Screen Savers' | grep .slideSaver", null).outputString, "Apple Slideshows"),
			new Array(widget.system("/bin/ls '/Library/Screen Savers'", null).outputString, "System Screen Savers")
		);
	}
	else
	{
		installedSavers = new Array(
			new Array("Arabesque\nFlurry\nRSS Visualizer\nRandom\nShell\nSpectrum\nWord of the Day\niTunes Artwork\n", ""),
			new Array("Abstract\nBeach\nCosmos\nForest\nNature Patterns\nPaper Shadow\n", "Apple Slideshows")
		);
	}

	// add each group of screen saver
	for (i = 0; i < installedSavers.length; i++)
		if (installedSavers[i][0] != null)
			w_appendScreenSavers(listElement, installedSavers[i][0].split("\n"), installedSavers[i][1]);
}


/*
 * Called when screen saver names need to be added to a <targetList>.
 *
 * User needs to specify the <targetList> in which the items need to be added,
 * and an array (<saverNames>) which contains the filenames of the screen savers.
 *
 * If a <groupLabel> is specified, all the items will be added under a group.
 */
function w_appendScreenSavers(targetList, saverNames, groupLabel)
{
	var i;
	var newGroup;
	var newOption;
	var saverName;
	var targetParent = targetList;

	if (groupLabel)
	{
		newGroup = document.createElement('optgroup');	
		newGroup.label = groupLabel;

		targetList.appendChild(document.createElement('option'));	// append an empty option for extra spacing between groups
		targetList.appendChild(newGroup);
		targetParent = newGroup;
	}

	// create each html <option> element and append to appropriate parent
	for (i = 0; i < saverNames.length-1; i++)
	{
		// remove extension if there is one
		saverName = (saverNames[i].indexOf('.') > 0) ? saverNames[i].substring(0, saverNames[i].indexOf('.')) : saverNames[i];

		newOption = document.createElement('option');
		newOption.value = saverName;
		newOption.text = saverName;

		targetParent.appendChild(newOption);
	}
}


/*
 * Retrieves the previously saved screen saver name if it exists.
 */
function w_loadSavedPrefs()
{
	var i;
	var listElement = document.getElementById("screenSaverList");
	var retrievedSaver = wr_getPreference(PREF_KEY_SAVER_NAME);

	if (retrievedSaver)
	{
		// selects the saved screen saver if it still exists in the list
		for (i = 0; i < listElement.options.length; i++)
		{
			if (retrievedSaver == listElement.options[i].value)
			{
				g_SaverName = retrievedSaver;
				listElement.selectedIndex = i;
				break;
			}
		}
	}
}


/*
 * Called when the screen saver name is selected from the back of the widget.
 * Sets the global variable to ensure the correct screen saver is executed when the time comes.
 * Saves the selection for later use[ and if it is currently running, update it to run the new value].
 */
function w_chooseSaver() 
{
	var selectedSaverName = this.options[this.selectedIndex].value;

	if (selectedSaverName != '')
	{
		g_SaverName = selectedSaverName;

		wr_setPreference(selectedSaverName, PREF_KEY_SAVER_NAME);
			
		//if (g_MyCommand)
		//{
		//	w_stopSaverEngine();
		//	w_runSaverEngine();
		//}
	}
}


/*
 * Runs the ScreenSaverEngine program on the desktop and applies the proper color to the button.
 * If a specific screen saver hasn't been chosen, then the default screen saver is used.
 * Once the command is initiated, it is controlled by the <g_MyCommand> global object.
 * This command is run asychronously.
 */
function w_runSaverEngine() 
{
	var command = "/System/Library/Frameworks/ScreenSaver.framework/Resources/ScreenSaverEngine.app/Contents/MacOS/ScreenSaverEngine -background";

	if (g_SaverName != "Default")
		command += " -module \"" + g_SaverName + "\"";

	if (window.widget)
		g_MyCommand = widget.system(command, w_commandEndHandler);
	else
		g_MyCommand = true;

	document.getElementById("execButton").style.background = "url(Images/button-color-red.png)";
}


/*
 * Kills the ScreenSaverEngine process and applies the proper color to the button.
 */
function w_stopSaverEngine()
{
	if (window.widget && g_MyCommand)
		g_MyCommand.cancel();

	document.getElementById("execButton").style.background = "url(Images/button-color-green.png)";
	
	g_MyCommand = null;
}


/*
 * Called once the command (in w_runSaverEngine()) completes its execution.
 */
function w_commandEndHandler(event)
{
	// By nature of this widget, this function should never be called since
	// the "ScreenSaverEngine" command will never finish execution on its own.
	// It terminates only when its execution is killed (in w_stopSaverEngine()).
}


/*
 * Called when a successful response has been received after checking for the version.
 * Sets the appropriate color of the version shown.
 * ✔ Green: Up-to-date.
 * ➜ Red: New version available (link to site).
 */
function w_versionQuery_response(latestVersion)
{
	if (latestVersion != null && latestVersion != '')
	{
		if ( g_CurrentVersion == latestVersion )
		{
			document.getElementById("version").style.color = "#51ad39"; // green
			document.getElementById("version").innerHTML = "✔&nbsp;<br />v" + g_CurrentVersion;
		}
		else
		{
			document.getElementById("version").style.color = "#c43311";	// red
			document.getElementById("version").innerHTML = "➜&nbsp;&nbsp;<br />v" + g_CurrentVersion;
			document.getElementById("version").onclick = function () { wr_openURL() };
		}
	}
	else
	{
		// empty page received
	}
}


/*
 * Called when the AJAX request has been initiated.
 * Changes the displayed text to indicate that something is happening in the background. 
 */
function w_versionQuery_startVisual()
{
	//document.getElementById("version").style.color = "yellow";
	document.getElementById("version").innerHTML = "v...";
}


/*
 * Called when the AJAX request has completed.
 * Restores the displayed text to the current version of the widget.
 */
function w_versionQuery_stopVisual()
{
	document.getElementById("version").innerHTML = "v" + g_CurrentVersion;
}


/*
 * Registers event handlers (including Dashboard-specific ones).
 */
function w_registerHandlers()
{
	g_ExecButton = new AppleButton(
		document.getElementById("execButton"),		//button element
		"",											//label
		38,											//height
		"Images/button-up-left.png",				//left image (up state)
		"Images/button-down-left.png",				//left image (down state)
		103,										//left image width
		"",											//middle image (up state)
		"",											//middle image (down state)
		"Images/button-up-right.png",				//right image (up state)
		"Images/button-down-right.png",				//right image (down state)
		15,											//right image width
		w_toggleSaver);								//onclick

	g_DoneButton = new AppleGlassButton(
		document.getElementById("doneButton"),		//button element
		wr_getLocalizedString("Done"),				//label
		w_toFront);									//onclick

	g_InfoButton = new AppleInfoButton(
		document.getElementById("infoButton"),		//button element
		document.getElementById("front"),			//element to activate when hovered
		"white",									//i color
		"white",									//rollie color
		w_toBack);									//onclick

	document.getElementById("logo").onclick = function () { wr_openURL() };
	document.getElementById("version").onclick = wr_versionQuery;
	document.getElementById("screenSaverList").onchange = w_chooseSaver;

	wr_widgetHandlers();
}


/*
 * Called when Widget instance is closed (not refreshed)
 */
function w_widgetRemovedHandler()
{
	// kill the ScreenSaverEngine Process in case it's still running
	w_stopSaverEngine();

	// remove preferences
	wr_setPreference(null, PREF_KEY_SAVER_NAME);
}


/*
 * Called when the Dashboard environment is activated
 */
function w_dashboardActivatedHandler()
{
	// start if not already running and isn't being installed for the first time...
	//if (!g_MyCommand && wr_getPreference(PREF_KEY_SAVER_NAME))
	//	w_runSaverEngine();
}
